/*
 * BCM53xx RoboSwitch utility functions
 *
 * Copyright 2004, Broadcom Corporation
 * All Rights Reserved.                
 *                                     
 * This is UNPUBLISHED PROPRIETARY SOURCE CODE of Broadcom Corporation;   
 * the contents of this file may not be disclosed to third parties, copied
 * or duplicated in any form, in whole or in part, without the prior      
 * written permission of Broadcom Corporation.                            
 *
 * $Id: etc_robo.c,v 1.8 2004/09/09 10:48:19 tallest Exp $
 */

#include <typedefs.h>
#include <osl.h>
#include <sbutils.h>

#include <bcmutils.h>
#include <bcmendian.h>
#include <proto/ethernet.h>
#include <bcmenetmib.h>
#include <et_dbg.h>
#include <etc.h>


#define VLAN_PAGE_BCM5325E 	0x34
#define VLAN_NUMVLANS		16	/* Max. # VLANs (VID starts from 0) */
#define PSEUDO_PHYAD		0x1E	/* MII Pseudo PHY address */
/* Misc. */
#define INVAL_DEVID	-1	/* invalid PHY id */
/* MII registers */
#define REG_MII_PAGE	0x10	/* MII Page register */
#define REG_MII_ADDR	0x11	/* MII Address register */
#define REG_MII_DATA0	0x18	/* MII Data register 0 */
#define REG_MII_DATA1	0x19	/* MII Data register 1 */
/* misc. constants */
#define MII_MAX_RETRY	100

/* Page numbers */
#define PAGE_CTRL	0x00	/* Control page */
#define PAGE_VLAN	0x34	/* VLAN page */

/* Control page registers */
#define REG_CTRL_MODE	0x0B	/* Switch Mode register */
#define REG_CTRL_MIIPO	0x0E	/* MII Port Override register */

/* VLAN page registers */
#define REG_VLAN_CTRL0	0x00	/* VLAN Control 0 register */
#define REG_VLAN_CTRL1	0x01	/* VLAN Control 1 register */
#define REG_VLAN_ACCESS	0x06	/* VLAN Table Access register */
#define REG_VLAN_WRITE	0x08	/* VLAN Write register */
#define REG_VLAN_READ	0x0C	/* VLAN Read register */
#define REG_VLAN_PTAG0	0x10	/* VLAN Default Port Tag register - port 0 */
#define REG_VLAN_PTAG1	0x12	/* VLAN Default Port Tag register - port 1 */
#define REG_VLAN_PTAG2	0x14	/* VLAN Default Port Tag register - port 2 */
#define REG_VLAN_PTAG3	0x16	/* VLAN Default Port Tag register - port 3 */
#define REG_VLAN_PTAG4	0x18	/* VLAN Default Port Tag register - port 4 */
#define REG_VLAN_PTAG5	0x1A	/* VLAN Default Port Tag register - MII port */
#define REG_VLAN_PMAP	0x20	/* VLAN Priority Re-map register */

#define GPIO_NUMPINS	16      /* # of GPIO pins */


/* Copy each token in wordlist delimited by space into word */
static int _strspn(const char * p, const char * s)
{
	int i,j;

	for (i=0;p[i];i++) {
		for (j=0;s[j];j++) {
			if (s[j] == p[i]) break;
		}
		if (!s[j]) break;
	}
	return(i);
}
static int _strcspn(char *p, char *s)
{
	int i,j;

	for (i=0;p[i];i++) {
		for (j=0;s[j];j++) {
			if (s[j] == p[i]) break;
		}
		if (s[j]) break;
	}
	return(i);
}
#define foreach(word, wordlist, next) \
	for (next = &wordlist[_strspn(wordlist, " ")], \
	     strncpy(word, next, sizeof(word)), \
	     word[_strcspn(word, " ")] = '\0', \
	     word[sizeof(word) - 1] = '\0', \
	     next = strchr(next, ' '); \
	     strlen(word); \
	     next = next ? &next[_strspn(next, " ")] : "", \
	     strncpy(word, next, sizeof(word)), \
	     word[_strcspn(word, " ")] = '\0', \
	     word[sizeof(word) - 1] = '\0', \
	     next = strchr(next, ' '))

     
	     
/* Private state per RoboSwitch */
typedef struct {
	void *sbh;			/* SiliconBackplane handle */
	void *vars;			/* variables handle */	
	uint32 devid;			/* device ID (phyid) */
	uint coreidx;			/* Current core index */
	uint32 ssl, clk, mosi, miso;	/* GPIO mapping */
	int cid, page;			/* Current chip ID and page */
		
	/* MII */
	void *ch;			/* etc */
	struct chops *chop;		/* etc chop */	
} robo_info_t;


/* Forward declarations */
//robo_info_t *
//robo_attach(void *sbh, uint32 ssl, uint32 clk, uint32 mosi, uint32 miso);
robo_info_t *
robo_attach(void *sbh, struct chops *chop, void *ch, uint32 ssl, uint32 clk, uint32 mosi, uint32 miso);

void robo_detach(robo_info_t *robo);
static void robo_enable(robo_info_t *robo);
void
robo_wreg(robo_info_t *robo, uint8 cid, uint8 page,
	  uint8 addr, void *buf, uint len);
void
robo_rreg(robo_info_t *robo, uint8 cid, uint8 page,
	  uint8 addr, void *buf, uint len);
void
robo_rvmii(robo_info_t *robo, uint8 cid , uint phyid); 

uint getgpiopin(char *vars, char *pin_name, uint def_pin);

int
mii_wreg(robo_info_t *robo, uint8 cid ,uint8 page,
          uint8 reg, void *val, int len);
          
int
mii_rreg(robo_info_t *robo, uint8 cid ,uint8 page,
          uint8 reg, void *val, int len);          

uint
getgpiopin(char *vars, char *pin_name, uint def_pin);

void
bcm53xx_reset(robo_info_t *robo);

int
bcm53xx_config(robo_info_t *robo, uint8 cid , char *vars ); 
void
bcm53xx_dump(robo_info_t *robo, uint8 cid  ); 



/* Return gpio pin number assigned to the named pin */
/*
* Variable should be in format:
*
*	gpio<N>=pin_name
*
* 'def_pin' is returned if there is no such variable found.
*/
uint
getgpiopin(char *vars, char *pin_name, uint def_pin)
{
	char name[] = "gpioXXXX";
	char *val;
	uint pin;

	/* Go thru all possibilities till a match in pin name */
	for (pin = 0; pin < GPIO_NUMPINS; pin ++) {
		sprintf(name, "gpio%d", pin);
		val = getvar(vars, name);
		if (val && !strcmp(val, pin_name))
			return pin;
	}
	return def_pin;
}


/* Get access to the RoboSwitch */
robo_info_t *
robo_attach(void *sbh, struct chops *chop, void *ch , uint32 ssl, uint32 clk, uint32 mosi, uint32 miso)
{
	robo_info_t *robo;

	/* Allocate private state */
	if (!(robo = MALLOC(sizeof(robo_info_t)))) {
		ET_ERROR(("robo_attach: out of memory"));
		return NULL;
	}
	bzero((char *) robo, sizeof(robo_info_t));
	robo->sbh = sbh;

	robo->cid = robo->page = -1;

	if (robo->devid == INVAL_DEVID) {
		
		/* 5325M Initialize GPIO outputs for  */
		robo->ssl = ssl;
		robo->clk = clk;
		robo->mosi = mosi;
		robo->miso = miso;		
		
		sb_gpioout(robo->sbh, robo->ssl | robo->clk | robo->mosi, robo->ssl);
		sb_gpioouten(robo->sbh, robo->ssl | robo->clk | robo->mosi | robo->miso, robo->ssl | robo->clk | robo->mosi);
		ET_5325E(("5325E\trobo_attach for 5325M\n" ));		
	}
	else {
		/* 5325E cache etc chop */
		robo->ch = ch;
		robo->chop = chop;	
		ET_5325E(("5325E\trobo_attach for 5325E\n" ));		
	}	
			
	ET_ERROR(("robo_attach: DONE\n")); //GREG	

	return robo;
}

/* Release access to the RoboSwitch */
void
robo_detach(robo_info_t *robo)
{
	/* Disable GPIO outputs */
	sb_gpioouten(robo->sbh, robo->ssl | robo->clk | robo->mosi, 0);

	/* Free private state */
	MFREE(robo, sizeof(robo_info_t));
}

/* Enable serial access to the chip */
static void
robo_enable(robo_info_t *robo)
{
	void *regs;

	/* Save current core index */
	robo->coreidx = sb_coreidx(robo->sbh);

	/* Switch to GPIO core for faster access */
	regs = sb_gpiosetcore(robo->sbh);
	ASSERT(regs);
}

/* Disable serial access to the chip */
static void
robo_disable(robo_info_t *robo)
{
	/* Switch back to original core */
	sb_setcoreidx(robo->sbh, robo->coreidx);
}
	
/* Write a byte stream to the chip */
static void
robo_write(robo_info_t *robo, uint8 *buf, uint len)
{
	uint i;
	uint8 mask;

	for (i = 0; i < len; i++) {
		/* Bit bang from MSB to LSB */
		for (mask = 0x80; mask; mask >>= 1) {
			/* Clock low */
			sb_gpioout(robo->sbh, robo->clk, 0);
			OSL_DELAY(10);

			/* Output on rising edge */
			if (mask & buf[i])
				sb_gpioout(robo->sbh, robo->mosi, robo->mosi);
			else
				sb_gpioout(robo->sbh, robo->mosi, 0);
		
			/* Clock high */
			sb_gpioout(robo->sbh, robo->clk, robo->clk);
			OSL_DELAY(10);
		}
	}
}

/* Handy macros for writing fixed length values */
#define robo_write8(robo, b) { uint8 val = (uint8) (b); robo_write((robo), &val, sizeof(val)); }
#define robo_write16(robo, w) { uint16 val = (uint16) (w); robo_write((robo), &val, sizeof(val)); }
#define robo_write32(robo, l) { uint32 val = (uint32) (l); robo_write((robo), &val, sizeof(val)); }

/* Read a byte stream from the chip */
static void
robo_read(robo_info_t *robo, uint8 *buf, uint len)
{
	uint i, timeout;
	uint8 rack, mask, byte;

	/* Timeout after 100 tries without RACK */
	for (i = 0, rack = 0, timeout = 100; i < len && timeout;) {
		/* Bit bang from MSB to LSB */
		for (mask = 0x80, byte = 0; mask; mask >>= 1) {
			/* Clock low */
			sb_gpioout(robo->sbh, robo->clk, 0);
			OSL_DELAY(10);

			/* Sample on rising edge */
			if (sb_gpioin(robo->sbh) & robo->miso)
				byte |= mask;

			/* Clock high */
			sb_gpioout(robo->sbh, robo->clk, robo->clk);
			OSL_DELAY(10);
		}

		/* RACK when bit 0 is high */
		if (!rack) {
			rack = (byte & 1);
			timeout--;
		} else {
			buf[i] = byte;
			i++;
		}
	}

	if (timeout == 0) {
		ET_ERROR(("robo_read: timeout"));
	}
}	

/* Handy macros for reading fixed length values */
#define robo_read8(robo) { uint8 val; robo_read((robo), &val, sizeof(val)); val; }
#define robo_read16(robo) { uint16 val; robo_read((robo), &val, sizeof(val)); val; }
#define robo_read32(robo) { uint32 val; robo_read((robo), &val, sizeof(val)); val; }

/* Select new chip and page */
static void
robo_select(robo_info_t *robo, uint8 cid, uint8 page)
{
	/* Chip and page already selected */
	if (robo->cid == (int) cid && robo->page == (int) page)
		return;
	robo->cid = (int) cid;
	robo->page = (int) page;

	/* Enable CS */
	sb_gpioout(robo->sbh, robo->ssl, 0);
	OSL_DELAY(10);

	/* Select new chip */
	robo_write8(robo, 0x61 | ((cid & 0x7) << 1));

	/* Select new page */
	robo_write8(robo, 0xff);
	robo_write8(robo, page);

	/* Disable CS */
	sb_gpioout(robo->sbh, robo->ssl, robo->ssl);
	OSL_DELAY(10);
}

/* Write chip register */
void
robo_wreg(robo_info_t *robo, uint8 cid, uint8 page, uint8 addr, void *buf, uint len)
{
	robo_enable(robo);

	/* Select chip and page */
	robo_select(robo, cid, page);

	/* Enable CS */
	sb_gpioout(robo->sbh, robo->ssl, 0);
	OSL_DELAY(10);

	/* Write */
	robo_write8(robo, 0x61 | ((cid & 0x7) << 1));
	robo_write8(robo, addr);
	robo_write(robo, buf, len);

	/* Disable CS */
	sb_gpioout(robo->sbh, robo->ssl, robo->ssl);
	OSL_DELAY(10);

	robo_disable(robo);
}

/* Read chip register */
void
robo_rreg(robo_info_t *robo, uint8 cid, uint8 page, uint8 addr, void *buf, uint len)
{
	robo_enable(robo);

	/* Select chip and page */
	robo_select(robo, cid, page);

	/* Enable CS */
	sb_gpioout(robo->sbh, robo->ssl, 0);
	OSL_DELAY(10);

	/* Fast read */
	robo_write8(robo, 0x10 | ((cid & 0x7) << 1));
	robo_write8(robo, addr);
 	robo_read(robo, buf, len);

	/* Disable CS */
	sb_gpioout(robo->sbh, robo->ssl, robo->ssl);
	OSL_DELAY(10);

	robo_disable(robo);
}




/* Enable reverse MII mode */
void
robo_rvmii(robo_info_t *robo, uint8 cid, uint phyid)
{
	uint8 mii;
	uint8 mii8;	
	uint8 value=0x10;
	
	/*
	   For BCM5325E-SPI switch have to set bits 7,3,2,1,0
	   In 5325M, bit 4 of MII port state override register directly drives RvMII enable signal.   
	*/
	
	if (robo->devid == INVAL_DEVID) {
		//5325M-SPI		
		/* MII port state override (page 0 register 14) */
		robo_rreg(robo, cid, 0, 14, &mii, sizeof(mii));
		ET_5325E(("5325E-SPI\t MII port state override=%X \n",mii));	
		/* Bit 4 enables reverse MII mode */
		mii |= value;
		robo_wreg(robo, cid, 0, 14, &mii, sizeof(mii));
		ET_5325E(("5325E-SPI\trobo_rvmii MII_REV mode= %X \n",mii));	
	
		/* Read back */
		robo_rreg(robo, cid, 0, 14, &mii, sizeof(mii));
		if (!(mii & 0x10)) {
			ET_ERROR(("robo_rvmii-SPI: error enabling mode"));
		}			
	}
	else {
		//5325E-MDIO	
		/* MII port state override (page 0 register 14) */
		mii_rreg           (robo, cid, PAGE_CTRL, REG_CTRL_MIIPO, &mii8, sizeof(mii8));
		ET_5325E(("5325E-MDIO\t Read Reg[%X]=%X\n",REG_CTRL_MIIPO,mii8 ));	
	
		/* Bit 4 enables reverse MII mode */
		if (!(mii8 & (1 << 4))) {
			/* Enable RvMII */
//			mii8 |= (1 << 4);
			mii8 = 0x9F; 	//In 5325E, bit 4 becomes effective only when bit 7 is set to 1.
			mii_wreg 	    (robo, cid ,PAGE_CTRL, REG_CTRL_MIIPO, &mii8, sizeof(mii8));
			ET_5325E(("5325E-MDIO\t Write Reg[%X]=%X\n",REG_CTRL_MIIPO,mii8 ));	
			
			/* Read back */
			mii_rreg           (robo, cid, PAGE_CTRL, REG_CTRL_MIIPO, &mii8, sizeof(mii8));
			if (!(mii8 & (1 << 4))) {
				ET_ERROR(("robo_rvmii over MDIO: enabling RvMII mode failed\n"));
			}
		}		
	}		

}


/* Configure the device and enable the VLANs */
int
bcm53xx_config(robo_info_t *robo, uint8 cid , char *vars )
{
	uint8 val8;
	uint16 val16;
	uint32 val32;
	
	/* port descriptor */
	struct {
		uint16 untag;	/* untag enable bit (Page 0x34 Address 0x08-0x0B Bit[11:6]) */
		uint16 member;	/* vlan member bit (Page 0x34 Address 0x08-0x0B Bit[5:0]) */
		uint8 pvidr;	/* pvid register address (Page 0x34 Address 0x10-0x1D) */
		uint8 cpu;	/* is this cpu port? */
	} pdesc[] = {
		/* port 0 */ {1 << 6, 1 << 0, 0x10, 0},
		/* port 1 */ {1 << 7, 1 << 1, 0x12, 0},
		/* port 2 */ {1 << 8, 1 << 2, 0x14, 0},
		/* port 3 */ {1 << 9, 1 << 3, 0x16, 0},
		/* port 4 */ {1 << 10, 1 << 4, 0x18, 0},
#if defined(PMON) || defined(_CFE_)
		/* mii port */ {1 << 11, 1 << 5, 0x1A, 1}, /*untag enable, only internal packets*/
#else
		/* mii port */ {0 << 11, 1 << 5, 0x1A, 1}, /*untag disable*/
#endif
	};
	uint16 vid;
	uint8 lan;

	
	/* enable 802.1Q vlan - VLAN Control 0 Register (Page 0x34, Address 0) */
	val8 = (1 << 7);
	val8 = val8 | 0x60; // tallest: support MAC clone;
	mii_wreg (robo, cid , 0x34, 0, &val8, sizeof(val8));
	/* setup each vlan */
	for (vid = 0; vid < VLAN_NUMVLANS; vid ++) {
		char vlanports[] = "vlanXXXXports";
		char port[] = "XXXX", *ports, *next;
		uint16 untag = 0;
		uint16 member = 0;

		/* get vlan member ports from nvram */
		sprintf(vlanports, "vlan%dports", vid);
		ports = getvar(vars, vlanports);
		ET_5325E(("5325E\tport=%s for %s\n",ports,vlanports));		
		if (!ports)
			continue;

		/* flag lan VLAN */
		lan = strchr(ports, '*') ? 1 : 0;
		
		/*
		* setup each port in the vlan. cpu port needs special handing 
		* (with or without output tagging) to support linux/pmon/cfe.
		*/
		foreach (port, ports, next) {
			int pid = bcm_atoi(port);
			
			/* make sure port # is within the range */
			if (pid >= sizeof(pdesc) / sizeof(pdesc[0])) {
				ET_ERROR(("port %d in vlan%dports is out of range\n", pid, vid));
				continue;
			}

			/* build VLAN registers values */
			untag |= pdesc[pid].untag;
			member |= pdesc[pid].member;
			
			/* set pvid - Default Port Tag Register (Page 0x34, Addres 0x10-0x1D) */
			if (!pdesc[pid].cpu || lan) {
				mii_wreg (robo, cid       ,0x34, pdesc[pid].pvidr, &vid, sizeof(vid));
			}				
		}

		/* setup VLAN ID and VLAN memberships */
		/* VLAN Write Register (Page 0x34, Address 0x08-0x0B) */
		val32 = (1 << 20)		/* valid write */
			| ((vid >> 4) << 12)	/* vlan id bit[11:4] */
			| untag			/* untag enable */
			| member		/* vlan members */
			;
		mii_wreg (robo, cid       ,0x34, 0x08, &val32, sizeof(val32));
		ET_5325E(("5325E\tWr[0x8]=%X\n",val32));
		
		/* VLAN Table Access Register (Page 0x34, Address 0x06-0x07) */
		val16 = (1 << 13) 	/* start command */
			| (1 << 12) 	/* write state */
			| vid		/* vlan id */
			;
		mii_wreg (robo, cid      , 0x34, 0x06, &val16, sizeof(val16));
		ET_5325E(("5325E\tWr[0x6]=%X\n",val16 ));
	}
		
	return 0;
}



/*
* Access switch registers through MII (MDC/MDIO) for 5325E
*/

/* Write register thru MDC/MDIO for 5325E */
int
mii_wreg(robo_info_t *robo, uint8 cid, uint8 page, uint8 reg, void *val, int len)
{
	uint16 cmd16, val16;
	int i;

	/* validate value length and buffer address */
	ASSERT (len == 1 || (len == 2 && !((int)val & 1)) || 
		(len == 4 && !((int)val & 3)));	

	/* set page number - MII register 0x10 */
	if (robo->page != page) {
		cmd16 = (page << 8)	/* page number */
			| 1		/* mdc/mdio access enable */
			;
		robo->chop->phywr(robo->ch, PSEUDO_PHYAD, REG_MII_PAGE, cmd16);
		robo->page = page;
	}
	/* write data - MII register 0x18-0x1B */
	switch (len) {
	case 1:
	case 2:
		val16 = (len == 1) ? *(uint8 *)val : *(uint16 *)val;
		robo->chop->phywr(robo->ch, PSEUDO_PHYAD, REG_MII_DATA0, val16);
		break;
	case 4:
		val16 = (uint16)*(uint32 *)val;
		robo->chop->phywr(robo->ch, PSEUDO_PHYAD, REG_MII_DATA0, val16);
		val16 = (uint16)(*(uint32 *)val >> 16);
		robo->chop->phywr(robo->ch, PSEUDO_PHYAD, REG_MII_DATA1, val16);
		break;
	}
	/* set register address - MII register 0x11 */
	cmd16 = (reg << 8)	/* register address */
		| 1		/* opcode write */
		;
	robo->chop->phywr(robo->ch, PSEUDO_PHYAD, REG_MII_ADDR, cmd16);
	/* is operation finished? */
	for (i = MII_MAX_RETRY; i > 0; i --) {
		val16 = robo->chop->phyrd(robo->ch, PSEUDO_PHYAD, REG_MII_ADDR);
		if ((val16 & 3) == 0)
			break;
	}
	/* timed out */
	if (!i) {
		ET_ERROR(("mii_wreg: timeout\n"));
		return -1;
	}
	return 0;
}

/* Read register thru MDC/MDIO */
int
mii_rreg(robo_info_t *robo, uint8 cid, uint8 page, uint8 reg, void *val, int len)
{
	uint16 cmd16, val16;
	int i;

	/* validate value length and buffer address */
	ASSERT (len == 1 || (len == 2 && !((int)val & 1)) || 
		(len == 4 && !((int)val & 3)));
	
	/* set page number - MII register 0x10 */
	if (robo->page != page) {
		cmd16 = (page << 8)	/* page number */
			| 1		/* mdc/mdio access enable */
			;
		robo->chop->phywr(robo->ch, PSEUDO_PHYAD, REG_MII_PAGE, cmd16);
		robo->page = page;
	}
	/* set register address - MII register 0x11 */
	cmd16 = (reg << 8)	/* register address */
		| 2		/* opcode read */
		;
	robo->chop->phywr(robo->ch, PSEUDO_PHYAD, REG_MII_ADDR, cmd16);
	/* is operation finished? */
	for (i = MII_MAX_RETRY; i > 0; i --) {
		val16 = robo->chop->phyrd(robo->ch, PSEUDO_PHYAD, REG_MII_ADDR);
		if ((val16 & 3) == 0)
			break;
	}
	/* timed out */
	if (!i) {
		ET_ERROR(("mii_rreg: timeout\n"));
		return -1;
	}
	/* read data - MII register 0x18-0x1B */
	switch (len) {
	case 1:
	case 2:
		val16 = robo->chop->phyrd(robo->ch, PSEUDO_PHYAD, REG_MII_DATA0);
		if (len == 1)
			*(uint8 *)val = (uint8)val16;
		else
			*(uint16 *)val = val16;
		break;
	case 4:
		val16 = robo->chop->phyrd(robo->ch, PSEUDO_PHYAD, REG_MII_DATA0);
		*(uint32 *)val = val16;
		val16 = robo->chop->phyrd(robo->ch, PSEUDO_PHYAD, REG_MII_DATA1);
		*(uint32 *)val += val16 << 16;
		break;
	}
	return 0;
}


/* 
  5325E/MDIO used GPIO pin 'robo_reset' to execute reset
  5325M/SPI  used h/w circuit to execute reset  
*/
void
bcm53xx_reset(robo_info_t *robo )
{
	uint32 reset;
	void *regs;
	
	/*
	* Reset sequence: RESET low(50ms)->high(20ms)
	*
	* We have to perform a full sequence for we don't know how long
	* it has been from power on till now.
	*/
	/* Trigger reset by nvram variable existance */
	reset = getgpiopin(robo->vars, "robo_reset", 0xff);	
	
	if ( reset != 0xff){	
	
		// nvram "robo_reset" exist , execute reset
		ET_5325E(("5325E\tGPIO reset GPIO=%d START\n",reset ));					
		
		reset = 1 << reset;
	
		/* Switch to GPIO core for faster access */
		robo->coreidx = sb_coreidx(robo->sbh);
		regs = sb_gpiosetcore(robo->sbh);
		ASSERT(regs);
	
		/* Keep RESET low for 50 ms */
		sb_gpioout(robo->sbh, reset, 0);
		sb_gpioouten(robo->sbh, reset, reset);
		bcm_mdelay(50);
		
		/* Keep RESET high for at least 20 ms */
		sb_gpioout(robo->sbh, reset, reset);
		bcm_mdelay(20);
		
		/* Switch back to original core */
		sb_setcoreidx(robo->sbh, robo->coreidx);
		
		ET_5325E(("5325E\tGPIO reset Done\n"));					
	}
	else  ET_ERROR(("GPIO robo_reset pin is NOT defined \n" ));			
}



void
bcm53xx_dump(robo_info_t *robo, uint8 cid  )
{
	uint8 val8;
	uint16 val16;
	uint32 val32;
	uint8 pvidr[6] = {0x10, 0x12, 0x14, 0x16, 0x18, 0x1A};
	int i;
	
	if (robo->devid == INVAL_DEVID) {		
		ET_5325E(("---------- 5325E robo switch over SPI dump ----------\n"));	
		
		/* Dump registers interested */
		robo_rreg          (robo, cid,  0, 0x0E, &val8, sizeof(val8));	
		
		ET_5325E(( "(0,0x0E)\tMII port state override regsiter: 0x%02x\n", val8));
		robo_rreg          (robo, cid,  0x34, 0, &val8, sizeof(val8));
		ET_5325E(("(0,0x0E)\tVLAN control 0 register: 0x%02x\n", val8));
		for (i = 0; i < VLAN_NUMVLANS; i ++) {
			val16 = (1 << 13)	/* start command */
				| i		/* vlan id */
				;
			robo_wreg(robo, cid      , 0x34, 0x06, &val16, sizeof(val16));		
			
	 		robo_rreg          (robo, cid,  0x34, 0x0C, &val32, sizeof(val32));
			
			ET_5325E(( "(0x34,0x0C)\tVLAN %d untag bits: 0x%02x member bits: 0x%02x\n",
				i, (val32 & 0x0FC0) >> 6, (val32 & 0x003F) ));
		}
		for (i = 0; i < 6; i ++) {
	 		robo_rreg          (robo, cid,  0x34, pvidr[i], &val16, sizeof(val16));	
			ET_5325E(("(0x34,0x%02x)\tPort %d VID: 0x%04x\n", pvidr[i], i, val16 ));
		}
	}
	else {
		ET_5325E(("---------- 5325E robo switch over MDIO dump ----------\n"));	
		
		/* Dump registers interested */
		mii_rreg	   (robo, cid,  0, 0x0E, &val8, sizeof(val8));		
		
		ET_5325E(( "(0,0x0E)\tMII port state override regsiter: 0x%02x\n", val8));
		mii_rreg          (robo, cid,  0x34, 0, &val8, sizeof(val8));
		ET_5325E(("(0,0x0E)\tVLAN control 0 register: 0x%02x\n", val8));
		for (i = 0; i < VLAN_NUMVLANS; i ++) {
			val16 = (1 << 13)	/* start command */
				| i		/* vlan id */
				;
			mii_wreg (robo, cid      , 0x34, 0x06, &val16, sizeof(val16));		
			
	 		mii_rreg           (robo, cid,  0x34, 0x0C, &val32, sizeof(val32));
			
			ET_5325E(( "(0x34,0x0C)\tVLAN %d untag bits: 0x%02x member bits: 0x%02x\n",
				i, (val32 & 0x0FC0) >> 6, (val32 & 0x003F) ));
		}
		for (i = 0; i < 6; i ++) {
			mii_rreg	   (robo, cid,  0x34, pvidr[i], &val16, sizeof(val16));		
			
			ET_5325E(("(0x34,0x%02x)\tPort %d VID: 0x%04x\n", pvidr[i], i, val16 ));
		}	
	}
}



