
#include <stdio.h>
#include <netdb.h>
#include <string.h>
#include <stdlib.h>
#include <getopt.h>
#include <ctype.h>

#include <iptables.h>

#include <linux/netfilter_ipv4/ipt_ipp2p.h>

#define IPP2P_VERSION "0.6"


static void
help(void)
{
    printf(
    "IPP2P v%s options:\n"
    " --ipp2p	Grab all known p2p packets\n"
    " --ipp2p-data	Grab all known p2p data packets\n"    
    " --edk		Grab all known eDonkey/eMule/Overnet packets\n"    
    " --edk-data	Grab all eDonkey/eMule/Overnet data packets\n"
    " --dc		Grab all known Direct Connect packets\n"
    " --dc-data	Grab all Direct Connect data packets\n"
    " --kazaa	Grab all KaZaA packets\n"    
    " --kazaa-data	Grab all KaZaA data packets\n"
    " --gnu		Grab all Gnutella packets\n"    
    " --gnu-data	Grab all Gnutella data packets\n"
    " --bit		Grab all BitTorrent packets\n"    
    " --apple	Grab all AppleJuice packets (beta - just a few tests until now)\n"
    " --soul		SoulSeek (beta - handle with care)\n"
    "\nNote that the follwing options will have the same meaning:\n"
    " '--ipp2p' is equal to '--edk --dc --kazaa --gnu'\n"
    " '--ipp2p-data' is equal to '--edk-data --dc-data --kazaa-data --gnu-data'\n"
    "\nIPP2P is designed for TCP only and has to be used together with -p tcp!\n"
    "\nExample: iptables -A FORWARD -p tcp -m ipp2p --ipp2p -j DROP\n\n"
    , IPP2P_VERSION);
}

							    

static struct option opts[] = {
        { "ipp2p", 0, 0, '1' },
        { "edk", 0, 0, '2' },	
        { "ipp2p-data", 0, 0, '3' },		
        { "kazaa-data", 0, 0, '4' },
        { "edk-data", 0, 0, '5' },	
        { "dc-data", 0, 0, '6' },
	{ "dc", 0, 0, '7' },
	{ "gnu-data", 0, 0, '8' },	
	{ "gnu", 0, 0, '9' },
	{ "kazaa", 0, 0, 'a' },
	{ "bit", 0, 0, 'b' },
	{ "apple", 0, 0, 'c' },	
	{ "soul", 0, 0, 'd' },	
        {0}
};

	

static void
init(struct ipt_entry_match *m, unsigned int *nfcache)
{
    *nfcache |= NFC_UNKNOWN;
}
	

static int
parse(int c, char **argv, int invert, unsigned int *flags,
	const struct ipt_entry *entry,
	unsigned int *nfcache,
	struct ipt_entry_match **match)
{
    struct ipt_p2p_info *info = (struct ipt_p2p_info *)(*match)->data;
    
    info->cmd = 0;
    switch (c) {
	case '1':		//cmd: ipp2p
	    if ((*flags & 1) == 1)
		    exit_error(PARAMETER_PROBLEM,
				"ipp2p: `--ipp2p' may only be "
				"specified once!");
	    if ((*flags & 4) == 4)
		    exit_error(PARAMETER_PROBLEM,
				"ipp2p: `--ipp2p-data' may only be "
				"specified alone!");
	    if ((*flags) != 0)
		    exit_error(PARAMETER_PROBLEM,
				"ipp2p: `--ipp2p' may only be "
				"specified alone!");
	    
	    if (invert) exit_error(PARAMETER_PROBLEM, "ipp2p: invert [!] is not allowed!");

	    *flags += 1;
	    info->cmd = *flags;
	    break;
	    
	case '2':		//cmd: edk
	    if ((*flags & 2) == 2)
		    exit_error(PARAMETER_PROBLEM,
				"ipp2p: `--edk' may only be "
				"specified once");
	    if ((*flags & 1) == 1)
		    exit_error(PARAMETER_PROBLEM,
				"ipp2p: `--ipp2p' may only be "
				"specified alone!");

	    if ((*flags & 4) == 4)
		    exit_error(PARAMETER_PROBLEM,
				"ipp2p: `--ipp2p-data' may only be "
				"specified alone!");

            if ((*flags & 16) == 16)
            exit_error(PARAMETER_PROBLEM,
                                "ipp2p: use `--edk' OR `--edk-data' but not both of them!");

	    if (invert) exit_error(PARAMETER_PROBLEM, "ipp2p: invert [!] is not allowed!");
	    
	    *flags += 2;
	    info->cmd = *flags;	    
	    break;

	case '3':		//cmd: ipp2p-data
	    if ((*flags & 4) == 4)
		    exit_error(PARAMETER_PROBLEM,
				"ipp2p: `--ipp2p-data' may only be "
				"specified once!");
	    if ((*flags & 1) == 1)
		    exit_error(PARAMETER_PROBLEM,
				"ipp2p: `--ipp2p' may only be "
				"specified alone!");
	    if ((*flags) != 0)
		    exit_error(PARAMETER_PROBLEM,
				"ipp2p: `--ipp2p-data' may only be "
				"specified alone!");


	    if (invert) exit_error(PARAMETER_PROBLEM, "ipp2p: invert [!] is not allowed!");

	    *flags += 4;
	    info->cmd = *flags;	    
	    break;

        case '4':		//cmd: kazaa-data
            if ((*flags & 8) == 8)
            exit_error(PARAMETER_PROBLEM,
                                "ipp2p: `--kazaa-data' may only be "
                                "specified once!");
	    if ((*flags & 1) == 1)
		    exit_error(PARAMETER_PROBLEM,
				"ipp2p: `--ipp2p' may only be "
				"specified alone!");
	    if ((*flags & 4) == 4)
		    exit_error(PARAMETER_PROBLEM,
				"ipp2p: `--ipp2p-data' may only be "
				"specified alone!");

            if ((*flags & 512) == 512)
            exit_error(PARAMETER_PROBLEM,
                                "ipp2p: use `--kazaa' OR `--kazaa-data' but not both of them!");
				
	    if (invert) exit_error(PARAMETER_PROBLEM, "ipp2p: invert [!] is not allowed!");

            *flags += 8;
	    info->cmd = *flags;
	    break;

	case '5':		//cmd: edk-data
            if ((*flags & 16) == 16)
            exit_error(PARAMETER_PROBLEM,
                                "ipp2p: `--edk-data' may only be "
                                "specified once!");
	    if ((*flags & 1) == 1)
		    exit_error(PARAMETER_PROBLEM,
				"ipp2p: `--ipp2p' may only be "
				"specified alone!");

	    if ((*flags & 4) == 4)
		    exit_error(PARAMETER_PROBLEM,
				"ipp2p: `--ipp2p-data' may only be "
				"specified alone!");

            if ((*flags & 2) == 2)
            exit_error(PARAMETER_PROBLEM,
                                "ipp2p: use `--edk' OR `--edk-data' but not both of them!");

	    if (invert) exit_error(PARAMETER_PROBLEM, "ipp2p: invert [!] is not allowed!");

            *flags += 16;
	    info->cmd = *flags;
	    break;																											

	case '6':		//cmd: dc-data
            if ((*flags & 32) == 32)
            exit_error(PARAMETER_PROBLEM,
                                "ipp2p: `--dc-data' may only be "
                                "specified once!");
	    if ((*flags & 4) == 4)
		    exit_error(PARAMETER_PROBLEM,
				"ipp2p: `--ipp2p-data' may only be "
				"specified alone!");

	    if ((*flags & 1) == 1)
		    exit_error(PARAMETER_PROBLEM,
				"ipp2p: `--ipp2p' may only be "
				"specified alone!");
            if ((*flags & 64) == 64)
            exit_error(PARAMETER_PROBLEM,
                                "ipp2p: use `--dc' OR `--dc-data' but not both of them!");

	    if (invert) exit_error(PARAMETER_PROBLEM, "ipp2p: invert [!] is not allowed!");

            *flags += 32;
	    info->cmd = *flags;
	    break;																											

	case '7':		//cmd: dc-data
            if ((*flags & 64) == 64)
            exit_error(PARAMETER_PROBLEM,
                                "ipp2p: `--dc' may only be "
                                "specified once!");
	    if ((*flags & 1) == 1)
		    exit_error(PARAMETER_PROBLEM,
				"ipp2p: `--ipp2p' may only be "
				"specified alone!");
	    if ((*flags & 4) == 4)
		    exit_error(PARAMETER_PROBLEM,
				"ipp2p: `--ipp2p-data' may only be "
				"specified alone!");

            if ((*flags & 32) == 32)
            exit_error(PARAMETER_PROBLEM,
                                "ipp2p: use `--dc' OR `--dc-data' but not both of them!");

	    if (invert) exit_error(PARAMETER_PROBLEM, "ipp2p: invert [!] is not allowed!");

            *flags += 64;
	    info->cmd = *flags;
	    break;																											


	case '8':		//cmd: gnu-data
            if ((*flags & 128) == 128)
            exit_error(PARAMETER_PROBLEM,
                                "ipp2p: `--gnu-data' may only be "
                                "specified once!");
	    if ((*flags & 1) == 1)
		    exit_error(PARAMETER_PROBLEM,
				"ipp2p: `--ipp2p' may only be "
				"specified alone!");
	    if ((*flags & 4) == 4)
		    exit_error(PARAMETER_PROBLEM,
				"ipp2p: `--ipp2p-data' may only be "
				"specified alone!");

            if ((*flags & 256) == 256)
            exit_error(PARAMETER_PROBLEM,
                                "ipp2p: use `--gnu' OR `--gnu-data' but not both of them!");

	    if (invert) exit_error(PARAMETER_PROBLEM, "ipp2p: invert [!] is not allowed!");

            *flags += 128;
	    info->cmd = *flags;
	    break;																											

	case '9':		//cmd: gnu
            if ((*flags & 256) == 256)
            exit_error(PARAMETER_PROBLEM,
                                "ipp2p: `--gnu' may only be "
                                "specified once!");
	    if ((*flags & 4) == 4)
		    exit_error(PARAMETER_PROBLEM,
				"ipp2p: `--ipp2p-data' may only be "
				"specified alone!");

	    if ((*flags & 1) == 1)
		    exit_error(PARAMETER_PROBLEM,
				"ipp2p: `--ipp2p' may only be "
				"specified alone!");
            if ((*flags & 128) == 128)
            exit_error(PARAMETER_PROBLEM,
                                "ipp2p: use `--gnu' OR `--gnu-data' but not both of them!");

	    if (invert) exit_error(PARAMETER_PROBLEM, "ipp2p: invert [!] is not allowed!");

            *flags += 256;
	    info->cmd = *flags;
	    break;																											

	case 'a':		//cmd: kazaa
            if ((*flags & 512) == 512)
            exit_error(PARAMETER_PROBLEM,
                                "ipp2p: `--kazaa' may only be "
                                "specified once!");
	    if ((*flags & 4) == 4)
		    exit_error(PARAMETER_PROBLEM,
				"ipp2p: `--ipp2p-data' may only be "
				"specified alone!");

	    if ((*flags & 1) == 1)
		    exit_error(PARAMETER_PROBLEM,
				"ipp2p: `--ipp2p' may only be "
				"specified alone!");
            if ((*flags & 8) == 8)
            exit_error(PARAMETER_PROBLEM,
                                "ipp2p: use `--kazaa' OR `--kazaa-data' but not both of them!");

	    if (invert) exit_error(PARAMETER_PROBLEM, "ipp2p: invert [!] is not allowed!");

            *flags += 512;
	    info->cmd = *flags;
	    break;																											

	case 'b':		//cmd: bit
            if ((*flags & 1024) == 1024)
            exit_error(PARAMETER_PROBLEM,
                                "ipp2p: `--bit' may only be "
                                "specified once!");
	    if (invert) exit_error(PARAMETER_PROBLEM, "ipp2p: invert [!] is not allowed!");

            *flags += 1024;
	    info->cmd = *flags;
	    break;																											

	case 'c':		//cmd: apple
            if ((*flags & 2048) == 2048)
            exit_error(PARAMETER_PROBLEM,
                                "ipp2p: `--apple' may only be "
                                "specified once!");
	    if (invert) exit_error(PARAMETER_PROBLEM, "ipp2p: invert [!] is not allowed!");

            *flags += 2048;
	    info->cmd = *flags;
	    break;																											


	case 'd':		//cmd: soul
            if ((*flags & 4096) == 4096)
            exit_error(PARAMETER_PROBLEM,
                                "ipp2p: `--soul' may only be "
                                "specified once!");
	    if (invert) exit_error(PARAMETER_PROBLEM, "ipp2p: invert [!] is not allowed!");

            *flags += 4096;
	    info->cmd = *flags;
	    break;																											


	default:
            exit_error(PARAMETER_PROBLEM,
	    "\nipp2p-parameter problem: for ipp2p usage type: iptables -m ipp2p --help\n");
	    return 0;
    }
    return 1;
}


static void
final_check(unsigned int flags)
{
    if (!flags)
            exit_error(PARAMETER_PROBLEM,
	    "\nipp2p-parameter problem: for ipp2p usage type: iptables -m ipp2p --help\n");
}



static void
print(const struct ipt_ip *ip,
        const struct ipt_entry_match *match,
	int numeric)
{
    struct ipt_p2p_info *info = (struct ipt_p2p_info *)match->data;
    
    printf("ipp2p v%s", IPP2P_VERSION);
    if ((info->cmd & 1) == 1) printf(" --ipp2p");
    if ((info->cmd & 4) == 4) printf(" --ipp2p-data");
    if ((info->cmd & 512) == 512) printf(" --kazaa");        
    if ((info->cmd & 8) == 8) printf(" --kazaa-data");
    if ((info->cmd & 128) == 128) printf(" --gnu-data");    
    if ((info->cmd & 256) == 256) printf(" --gnu");
    if ((info->cmd & 2) == 2) printf(" --edk");    
    if ((info->cmd & 16) == 16) printf(" --edk-data");    
    if ((info->cmd & 32) == 32) printf(" --dc-data");
    if ((info->cmd & 64) == 64) printf(" --dc");    
    if ((info->cmd & 1024) == 1024) printf(" --bit");        
    if ((info->cmd & 2048) == 2048) printf(" --apple");    
    if ((info->cmd & 4096) == 4096) printf(" --soul");        
    printf(" ");
}
			    						   


static void
save(const struct ipt_ip *ip, const struct ipt_entry_match *match)
{
    struct ipt_p2p_info *info = (struct ipt_p2p_info *)match->data;
    
    if ((info->cmd & 1) == 1) printf("--ipp2p ");
    if ((info->cmd & 4) == 4) printf("--ipp2p-data ");
    if ((info->cmd & 512) == 512) printf("--kazaa ");        
    if ((info->cmd & 8) == 8) printf("--kazaa-data ");
    if ((info->cmd & 128) == 128) printf("--gnu-data ");    
    if ((info->cmd & 256) == 256) printf("--gnu ");
    if ((info->cmd & 2) == 2) printf("--edk ");    
    if ((info->cmd & 16) == 16) printf("--edk-data ");    
    if ((info->cmd & 32) == 32) printf("--dc-data ");
    if ((info->cmd & 64) == 64) printf("--dc ");    
    if ((info->cmd & 1024) == 1024) printf("--bit ");        
    if ((info->cmd & 2048) == 2048) printf("--apple ");    
    if ((info->cmd & 4096) == 4096) printf("--soul ");        
}

		


static
struct iptables_match ipp2p
= { NULL,
    "ipp2p",
    "1.2.11", //NETFILTER_VERSION, 
    IPT_ALIGN(sizeof(struct ipt_p2p_info)),
    IPT_ALIGN(sizeof(struct ipt_p2p_info)),
    &help,
    &init,
    &parse,
    &final_check,
    &print,
    &save,
    opts
};
					    


void _init(void)
{
    register_match(&ipp2p);
}

 

