/*
 * Misc initialization and support routines for self-booting
 * compressed image.
 *
 * Copyright 2001-2003, Broadcom Corporation
 * All Rights Reserved.
 * 
 * THIS SOFTWARE IS OFFERED "AS IS", AND BROADCOM GRANTS NO WARRANTIES OF ANY
 * KIND, EXPRESS OR IMPLIED, BY STATUTE, COMMUNICATION OR OTHERWISE. BROADCOM
 * SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A SPECIFIC PURPOSE OR NONINFRINGEMENT CONCERNING THIS SOFTWARE.
 *
 * $Id: misc.c,v 1.1.1.6 2003/11/08 08:11:32 honor Exp $
 */

#include <linux/autoconf.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/mm.h>
#include <linux/serial_reg.h>
#include <linux/serial.h>
#include <linux/delay.h>

#include <asm/bootinfo.h>
#include <asm/cpu.h>
#include <asm/bcache.h>
#include <asm/io.h>
#include <asm/page.h>
#include <asm/pgtable.h>
#include <asm/system.h>
#include <asm/mmu_context.h>

#include <typedefs.h>
#include <bcmdevs.h>
#include <bcmnvram.h>
#include <bcmutils.h>
#include <sbconfig.h>
#include <sbextif.h>
#include <sbchipc.h>
#include <sbmips.h>
#include <sbmemc.h>
#include <sflash.h>
#include <osl.h>
#include <sbutils.h>

/* At 125 MHz */
unsigned long loops_per_jiffy = 625000;

static void *sbh;
static chipcregs_t *cc;
static struct serial_struct uart;
static struct sflash *sflash;

#define LOG_BUF_LEN	(1024)
#define LOG_BUF_MASK	(LOG_BUF_LEN-1)
static char log_buf[LOG_BUF_LEN];
static unsigned long log_start;

/* Declarations needed for the cache related includes below */

/* Primary cache parameters. These declarations are needed*/
static int icache_size, dcache_size;	/* Size in bytes */
static int ic_lsize, dc_lsize;		/* LineSize in bytes */

#include <asm/cacheops.h>
#include <asm/bcm4710_cache.h>

__BUILD_SET_C0(taglo,CP0_TAGLO);
__BUILD_SET_C0(taghi,CP0_TAGHI);

static void
cache_init(void)
{
	unsigned int config1;
	unsigned int sets, ways;
	unsigned int start, end;

	config1 = read_c0_config1(); 

	/* Instruction Cache Size = Associativity * Line Size * Sets Per Way */
	if ((ic_lsize = ((config1 >> 19) & 7)))
		ic_lsize = 2 << ic_lsize;
	sets = 64 << ((config1 >> 22) & 7);
	ways = 1 + ((config1 >> 16) & 7);
	icache_size = ic_lsize * sets * ways;

	start = KSEG0;
	end = (start + icache_size);
	clear_c0_taglo(~0);
	clear_c0_taghi(~0);
	while (start < end) {
		cache_unroll(start, Index_Store_Tag_I);
		start += ic_lsize;
	}

	/* Data Cache Size = Associativity * Line Size * Sets Per Way */
	if ((dc_lsize = ((config1 >> 10) & 7)))
		dc_lsize = 2 << dc_lsize;
	sets = 64 << ((config1 >> 13) & 7);
	ways = 1 + ((config1 >> 7) & 7);
	dcache_size = dc_lsize * sets * ways;

	start = KSEG0;
	end = (start + dcache_size);
	clear_c0_taglo(~0);
	clear_c0_taghi(~0);
	while (start < end) {
		cache_unroll(start, Index_Store_Tag_D);
		start += dc_lsize;
	}
}

#ifdef SIM

#define serial_in(info, offset) 0
#define serial_out(info, offset, value) do {} while (0)
#define putc(c) do {} while (0)
#define puts(cs) do {} while (0)
#define puthex(c) do {} while (0)
#define putdec(cs) do {} while (0)

#else

static inline unsigned int
serial_in(struct serial_struct *info, int offset)
{
#ifdef CONFIG_BCM4310
	readb((unsigned long) info->iomem_base +
	      (UART_SCR<<info->iomem_reg_shift));
#endif
	return readb((unsigned long) info->iomem_base +
		     (offset<<info->iomem_reg_shift));
}

static inline void
serial_out(struct serial_struct *info, int offset, int value)
{
	writeb(value, (unsigned long) info->iomem_base +
	       (offset<<info->iomem_reg_shift));
#ifdef CONFIG_BCM4704
	*((volatile unsigned int *) KSEG1ADDR(0x18000000));
#endif
}

static void
putc(int c)
{
	/* CR before LF */
	if (c == '\n')
		putc('\r');

	/* Store in log buffer */
	*((char *) KSEG1ADDR(&log_buf[log_start])) = (char) c;
	log_start = (log_start + 1) & LOG_BUF_MASK;

	while (!(serial_in(&uart, UART_LSR) & UART_LSR_THRE));
	serial_out(&uart, UART_TX, c);
}

static void
puts(const char *cs)
{
	char *s = (char *) cs;
	short c;
	
	while (1) {
		c = *(short *)(s);
		if ((char)(c & 0xff))
			putc((char)(c & 0xff));
		else
			break;
		if ((char)((c >> 8) & 0xff))
			putc((char)((c >> 8) & 0xff));
		else
			break;
		s += sizeof(short);
	}
}

static void
puthex(unsigned int h)
{
	char c;
	int i;
	
	for (i = 7; i >= 0; i--) {
		c = (char)((h >> (i * 4)) & 0xf);
		c += (c > 9) ? ('a' - 10) : '0';
		putc(c);
	}
}

void
putdec(int d)
{
	int leading_zero;
	unsigned int divisor, result, remainder;

	if (d < 0) {
		d = -d;
		putc('-');
	}

	leading_zero = 1;
	remainder = d;

	for (divisor = 1000000000; 
	     divisor > 0; 
	     divisor /= 10) {
		result = remainder / divisor;
		remainder %= divisor;

		if (result != 0 || divisor == 1)
			leading_zero = 0;

		if (leading_zero == 0)
			putc((char)(result) + '0');
	}
}

#endif /* !SIM */

static void
reset_usb(void)
{
#if defined(CONFIG_USB_OHCI) || defined(CONFIG_USB_OHCI_MODULE) || defined(CONFIG_USBDEV) || defined(CONFIG_USBDEV_MODULE)
	void *usb;

#if defined(CONFIG_USBDEV) || defined(CONFIG_USBDEV_MODULE)
	if ((sb_setcore(sbh, SB_USB11D, 0)) &&
	    (sb_corerev(sbh) == 0) &&
	    (sb_mips_clock(sbh) != 192000000))
		sb_mips_setclock(sbh, 192000000, 96000000, 0);
#endif

	if (!(usb = sb_setcore(sbh, SB_USB, 0)))
		return;

	if ((readl(&cc->intstatus) & 0x80000000) == 0 && (sb_corerev(sbh) == 1)) {
		/* Reset USB host core into sane state */
		sb_coreflags(sbh, (1 << 29), (1 << 29));
		udelay(10);
		sb_coreflags(sbh, (1 << 29), 0);
		udelay(10);
		/* Reset backplane to 96 MHz */
		sb_mips_setclock(sbh, 96000000, 0, 0);
	}
#endif
}

#define INBUFSIZ 4096
#define WSIZE 0x8000    /* window size--must be a power of two, and */
			/*  at least 32K for zip's deflate method */

static uchar inbuf[INBUFSIZ];	/* input buffer */
static ulong insize;		/* valid bytes in inbuf */
static ulong inptr;		/* index of next byte to be processed in inbuf */

static uchar *outbuf;		/* output buffer */
static ulong bytes_out;		/* valid bytes in outbuf */

static ulong inoff;		/* offset of input data */

static void
error(char *x)
{
	puts("\n\n");
	puts(x);
	puts("\n\n -- System halted");

	for (;;);
}

static ulong free_mem_ptr;
static ulong free_mem_ptr_end;

static void *
malloc(int size)
{
	void *p;

	/* Sanity check */
	if (size < 0)
		error("Malloc error");
	if (free_mem_ptr <= 0)
		error("Memory error");

	/* Align */
	free_mem_ptr = (free_mem_ptr + 3) & ~3;

	p = (void *) free_mem_ptr;
	free_mem_ptr += size;

	if (free_mem_ptr >= free_mem_ptr_end)
		error("Out of memory");

	return p;
}

static void
free(void *where)
{
}

static int
fill_inbuf(void)
{
	int bytes;

	for (insize = 0; insize < INBUFSIZ; insize += bytes, inoff += bytes) {
		if (sflash) {
			if ((bytes = sflash_read(cc, inoff, INBUFSIZ - insize, &inbuf[insize])) < 0)
				return bytes;
		} else {
			*((uint32 *) &inbuf[insize]) = *((uint32 *) KSEG1ADDR(0x1fc00000 + inoff));
			bytes = sizeof(uint32);
		}
	}

	inptr = 1;

	return inbuf[0];
}

#if defined(USE_GZIP)

/*
 * gzip declarations
 */

#define OF(args) args
#define STATIC static

#define memzero(s, n)	memset ((s), 0, (n))

typedef unsigned char  uch;
typedef unsigned short ush;
typedef unsigned long  ulg;

#define get_byte()  (inptr < insize ? inbuf[inptr++] : fill_inbuf())

/* Diagnostic functions (stubbed out) */

#define Assert(cond,msg)
#define Trace(x)
#define Tracev(x)
#define Tracevv(x)
#define Tracec(c,x)
#define Tracecv(c,x)

static uch window[WSIZE];	/* Sliding window buffer */
static unsigned outcnt;		/* bytes in window buffer */

static void
gzip_mark(void **ptr)
{
	*ptr = (void *) free_mem_ptr;
}

static void
gzip_release(void **ptr)
{
	free_mem_ptr = (long) *ptr;
}

static void flush_window(void);

#include "../../../../../lib/inflate.c"

/* ===========================================================================
 * Write the output window window[0..outcnt-1] and update crc and bytes_out.
 * (Used for the decompressed data only.)
 */
static void
flush_window(void)
{
	ulg c = crc;
	unsigned n;
	uch *in, *out, ch;

	in = window;
	out = &outbuf[bytes_out];
	for (n = 0; n < outcnt; n++) {
		ch = *out++ = *in++;
		c = crc_32_tab[((int)c ^ ch) & 0xff] ^ (c >> 8);
	}
	crc = c;
	bytes_out += (ulg)outcnt;
	outcnt = 0;
	puts(".");
}

#elif defined(USE_BZIP2)

/*
 * bzip2 declarations
 */

#include "../../../../../lib/bzip2_inflate.c"

static int
bunzip2(void)
{
	bz_stream bzstream;
	int ret = 0;

        bzstream.bzalloc = 0;
        bzstream.bzfree = 0;
	bzstream.opaque = 0;
	bzstream.avail_in = 0;

        if ((ret = BZ2_bzDecompressInit(&bzstream, 0, 1)) != BZ_OK)
		return ret;

	for (;;) {
		if (bzstream.avail_in == 0) {
			fill_inbuf();
			bzstream.next_in = inbuf;
			bzstream.avail_in = insize;
		}
		bzstream.next_out = &outbuf[bytes_out];
		bzstream.avail_out = WSIZE;
	 	if ((ret = BZ2_bzDecompress(&bzstream)) != BZ_OK)
			break;
		bytes_out = bzstream.total_out_lo32;
		puts(".");
	}

	if (ret == BZ_STREAM_END)
		ret = BZ2_bzDecompressEnd(&bzstream);

	if (ret == BZ_OK)
		ret = 0;

	return ret;
}

#endif

extern char input_data[];
extern int input_len;
extern char text_start[], text_end[];
extern char data_start[], data_end[];
extern char bss_start[], bss_end[];

static void
load(void)
{
	int ret = 0;

	/* Offset from beginning of flash */
	inoff = (ulong) input_data - (ulong) text_start;
	outbuf = (uchar *) LOADADDR;
	free_mem_ptr = free_mem_ptr_end = (ulong) bss_end;

#if defined(USE_GZIP)
	free_mem_ptr_end += 0x100000;
	puts("Decompressing...");
	makecrc();
	ret = gunzip();
#elif defined(USE_BZIP2)
	/* Small decompression algorithm uses up to 2300k of memory */
	free_mem_ptr_end += 0x300000;
	puts("Decompressing...");
	ret = bunzip2();
#else
	puts("Copying...");
	while (bytes_out < input_len) {
		fill_inbuf();
		memcpy(&outbuf[bytes_out], inbuf, insize);
		bytes_out += insize;
	}
#endif
	if (ret) {
		puts("error ");
		putdec(ret);
	} else
		puts("done\n");
}

static void
sflash_self(chipcregs_t *cc)
{
	unsigned char *start = text_start;
	unsigned char *end = data_end;
	unsigned char *cur = start;
	unsigned int erasesize, len;

	while (cur < end) {
		/* Erase sector */
		puts("Erasing sector 0x");
		puthex(cur - start);
		puts("...");
		if ((erasesize = sflash_erase(cc, cur - start)) < 0) {
			puts("error\n");
			break;
		}
		while (sflash_poll(cc, cur - start));
		puts("done\n");

		/* Write sector */
		puts("Writing sector 0x");
		puthex(cur - start);
		puts("...");
		while (erasesize) {
			if ((len = sflash_write(cc, cur - start, erasesize, cur)) < 0)
				break;
			while (sflash_poll(cc, cur - start));
			cur += len;
			erasesize -= len;
		}
		if (erasesize) {
			puts("error\n");
			break;
		}
		puts("done\n");
	}
}

static void
_change_cachability(u32 cm)
{
	u32 prid;

	change_c0_config(CONF_CM_CMASK, cm);
	prid = read_c0_prid();
	if ((prid & (PRID_COMP_MASK | PRID_IMP_MASK)) ==
	    (PRID_COMP_BROADCOM | PRID_IMP_BCM3302)) {
		cm = read_c0_diag();
		/* Enable icache */
		cm |= (1 << 31);
		/* Enable dcache */
		cm |= (1 << 30);
		write_c0_diag(cm);
	}
}	
static void (*change_cachability)(u32);

static void
serial_add(void *regs, uint irq, uint baud_base, uint reg_shift)
{
	int quot;

	if (uart.iomem_base)
		return;

	uart.iomem_base = regs;
	uart.irq = irq;
	uart.baud_base = baud_base / 16;
	uart.iomem_reg_shift = reg_shift;

	/* Set baud and 8N1 */
	quot = uart.baud_base / 115200;
	serial_out(&uart, UART_LCR, UART_LCR_DLAB);
	serial_out(&uart, UART_DLL, quot & 0xff);
	serial_out(&uart, UART_DLM, quot >> 8);
	serial_out(&uart, UART_LCR, UART_LCR_WLEN8);

	/* According to the Synopsys website: "the serial clock
	 * modules must have time to see new register values
	 * and reset their respective state machines. This
	 * total time is guaranteed to be no more than
	 * (2 * baud divisor * 16) clock cycles of the slower
	 * of the two system clocks. No data should be transmitted
	 * or received before this maximum time expires."
	 */
	OSL_DELAY(1000);
}

void
c_main(unsigned long ra)
{
	/* Disable interrupts */
	clear_c0_status(1);

	/* Scan backplane */
	sbh = sb_kattach();
	cc = sb_setcore(sbh, SB_CC, 0);

	if (cc && PHYSADDR(ra) >= 0x1fc00000)
		reset_usb();

	sb_mips_init(sbh);
	sb_serial_init(sbh, serial_add);

	/* Must be in KSEG1 to change cachability */
	cache_init();
	change_cachability = (void (*)(u32)) KSEG1ADDR((unsigned long)(_change_cachability));
	change_cachability(CONF_CM_CACHABLE_NONCOHERENT);

	/* Initialize serial flash */
	sflash = cc ? sflash_init(cc) : NULL;

	/* Copy self to flash if we booted from SDRAM */
	if (PHYSADDR(ra) < 0x1fc00000) {
		if (sflash)
			sflash_self(cc);
	}

	/* Load binary */
	load();

	/* Flush all caches */
	blast_dcache();
	blast_icache();

	/* Jump to load address */
	((void (*)(void)) LOADADDR)();
}

/* asm/io.h */

void * __ioremap(phys_t phys_addr, phys_t size, unsigned long flags)
{
	return (void *) KSEG1ADDR(phys_addr);
}

void iounmap(void *addr)
{
}

/* linux/slab.h */

void * kmalloc (size_t size, int flags)
{
	return malloc(size);
}

void kfree(const void *objp)
{
	free((void *) objp);
}

/* bcmnvram.h */

char *
nvram_get(const char *name)
{
	return NULL;
}

/* osl.h */

uint32
osl_pci_read_config(void *loc, uint offset, uint size)
{
	return (0xffffffff);
}

void
osl_pci_write_config(void *loc, uint offset, uint size, uint val)
{
}

void
osl_pcmcia_read_attr(void *osh, uint offset, void *buf, int size)
{
	ASSERT(0);
}

void
osl_pcmcia_write_attr(void *osh, uint offset, void *buf, int size)
{
	ASSERT(0);
}

void
osl_assert(char *exp, char *file, int line)
{
}

/* bcmutils.h */

char*
getvar(char *vars, char *name)
{
	return NULL;
}

int
getintvar(char *vars, char *name)
{
	return 0;
}

void
bcm_mdelay(uint ms)
{
	uint i;

	for (i = 0; i < ms; i++) {
		OSL_DELAY(1000);
	}
}

/* bcmsrom.h */

int
srom_var_init(uint bus, void *curmap, void *osh, char **vars, int *count)
{
	return 0;
}

int
srom_read(uint bus, void *curmap, void *osh, uint byteoff, uint nbytes, uint16 *buf)
{
	return 0;
}

/* string.h */

int sprintf(char * buf, const char *fmt, ...)
{
	return 0;
}

asmlinkage int printk(const char *fmt, ...)
{
	return 0;
}

void*
memset(void* s, int c, size_t n)
{
        int i;
        char *ss = (char*)s;

        for (i=0;i<n;i++) ss[i] = c;
        return s;
}

void*
memcpy(void* __dest, __const void* __src, size_t __n)
{
	int i;
	char *d = (char *)__dest, *s = (char *)__src;

	for (i=0;i<__n;i++) d[i] = s[i];
	return __dest;
}
